import { FastifyInstance } from "fastify";
import CategoriaUseCase from "../userCases/categoria.usercases";
import { Categoria } from "../interfaces/categoria.interfaces";
import { authMiddleware } from "../middlewares/auth.middleware";
import { z } from 'zod';

export async function categoriaRoutes(fastify: FastifyInstance) {
    const categoriaUserCase = new CategoriaUseCase();

    fastify.post<{ Body: Categoria }>('/register', { preHandler: authMiddleware as any }, async (req, reply) => {
        const createCategoriaSchema = z.object({
            nome: z.string(),
            descricao: z.string(),
        })

        const { nome, descricao } = createCategoriaSchema.parse(req.body);

        try {
            const data = await categoriaUserCase.create({ nome, descricao })
            return reply.send({ message: "Registro realizado com sucesso", data });
        } catch (error) {
            reply.status(500).send(error)
        }
    })

    fastify.get('/all', async (req, reply) => {
        try {
            const data = await categoriaUserCase.findAll()
            return reply.send(data);
        } catch (error) {
            reply.status(500).send(error)
        }
    })

    fastify.get<{ Params: { id: number } }>('/:id', async (req, reply) => {
        const { id } = req.params
        try {
            const data = await categoriaUserCase.findById(Number(id))
            return reply.send(data);
        } catch (error) {
            reply.status(500).send(error)
        }
    })

    fastify.put<{ Body: Categoria }>('/update', { preHandler: authMiddleware as any }, async (req, reply) => {
        const { id, nome, descricao } = req.body
        try {
            const data = await categoriaUserCase.update({ id, nome, descricao })
            return reply.send({ message: "Registro atualizado com sucesso", data });
        } catch (error) {
            reply.status(500).send(error)
        }
    })

    fastify.delete<{ Params: { id: number } }>('/:id', { preHandler: authMiddleware as any }, async (req, reply) => {
        const { id } = req.params
        try {
            await categoriaUserCase.delete(Number(id))
            return reply.send({ message: "Registro deletado com sucesso" });
        } catch (error) {
            reply.status(500).send(error)
        }
    })
};