"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AuthRepositoryPrisma = void 0;
const prisma_client_1 = require("../database/prisma.client");
const bcrypt_1 = __importDefault(require("bcrypt"));
const jsonwebtoken_1 = __importDefault(require("jsonwebtoken"));
const crypto_1 = __importDefault(require("crypto"));
class AuthRepositoryPrisma {
    async create(data) {
        const hashedPassword = await bcrypt_1.default.hash(data.password, 10);
        const result = await prisma_client_1.prisma.user.create({
            data: {
                email: data.email,
                password: hashedPassword,
                role: data.role,
                cliente: data.cliente ? {
                    create: data.cliente
                } : undefined,
                profissional: data.profissional ? {
                    create: data.profissional
                } : undefined
            },
            include: {
                cliente: true,
                profissional: true
            }
        });
        const { password, ...userWithoutPassword } = result;
        return userWithoutPassword;
    }
    async login(data) {
        const user = await prisma_client_1.prisma.user.findUnique({
            where: {
                email: data.email
            }
        });
        if (!user) {
            throw new Error('Usuário não encontrado');
        }
        const isPasswordValid = await bcrypt_1.default.compare(data.password, user.password);
        if (!isPasswordValid) {
            throw new Error('Senha incorreta');
        }
        const { password, ...userWithoutPassword } = user;
        const token = jsonwebtoken_1.default.sign({ id: user.id }, process.env.JWT_PASS ?? "", { expiresIn: '1h' });
        return { ...userWithoutPassword, token };
    }
    ;
    async forgotPassword(email) {
        const user = await prisma_client_1.prisma.user.findUnique({
            where: {
                email
            }
        });
        if (!user) {
            throw new Error('Usuário não encontrado');
        }
        const token = crypto_1.default.randomBytes(8).toString("hex");
        const now = new Date();
        now.setHours(now.getHours() + 1);
        const existingToken = await prisma_client_1.prisma.authToken.findFirst({
            where: {
                userId: user.id,
            },
        });
        if (existingToken) {
            // Token exists, update it
            await prisma_client_1.prisma.authToken.update({
                where: {
                    id: existingToken.id,
                },
                data: {
                    passwordResetToken: token,
                    passwordResetExpires: now,
                },
            });
        }
        else {
            // Token does not exist, create it
            await prisma_client_1.prisma.authToken.create({
                data: {
                    userId: user.id,
                    passwordResetToken: token,
                    passwordResetExpires: now,
                },
            });
        }
        /*
        enviarEmail(user.email, "Reset Password ", 'esqueciasenha', {
            dados: {
                nome: user.name,
                conteudo: `http://localhost:3000/reset-password?token=${token}`,
                assunto: "Reset Password ",
                operatingSystem: "",
                browserName: "",
            }
        });
        */
    }
    async resetPassword(data) {
        const { token, password, email } = data;
        const tokenData = await prisma_client_1.prisma.authToken.findFirst({
            where: {
                passwordResetToken: token,
            },
        });
        if (!tokenData) {
            throw new Error('Token inválido');
        }
        const user = await prisma_client_1.prisma.user.findUnique({
            where: {
                id: tokenData.userId,
            },
        });
        if (!user) {
            throw new Error('Usuário não encontrado');
        }
        const hashedPassword = await bcrypt_1.default.hash(password, 10);
        await prisma_client_1.prisma.user.update({
            where: {
                id: user.id,
            },
            data: {
                password: hashedPassword,
            },
        });
    }
}
exports.AuthRepositoryPrisma = AuthRepositoryPrisma;
