import { FastifyReply, FastifyRequest } from 'fastify';
import jwt from "jsonwebtoken";
import { prisma } from "../database/prisma.client";
import { UnauthorizedError } from '../exceptions/api.error';

type JwtPayload = {
    id: number;
    exp: number;
};

interface CustomRequest extends FastifyRequest {
    user: any;
}

export async function authMiddleware(req: CustomRequest, reply: FastifyReply) {
    try {
        // Verificar se o token está presente no cabeçalho da solicitação
        const authHeader = req.headers.authorization;

        if (!authHeader) {
            return reply.status(401).send({ message: 'Token não fornecido' });
        }

        const token = authHeader.split(' ')[1];
        if (!token) {
            return reply.status(401).send({ message: 'Token malformado' });
        }

        // Decodificar e verificar o token JWT
        const secret = process.env.JWT_PASS ?? "";
        if (!secret) {
            throw new Error("JWT_PASS não está definido");
        }

        // Decodificar e verificar o token JWT
        const payload = jwt.verify(token, secret) as JwtPayload;

        const { id, exp } = payload;
        if (Date.now() >= exp * 1000) {
            throw new UnauthorizedError("Token expirado");
        }
        // Buscar o usuário no banco de dados
        const user = await prisma.user.findUnique({ where: { id: id } });


        if (!user) {
            return reply.status(401).send({ message: 'Usuário não encontrado' });
        }
        req.user = { ...user, password: undefined };

    } catch (error) {
        return reply.status(401).send({ message: 'Autenticação falhou', error });
    }
}